//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

package main

import (
	"database/sql"
	"flag"
	"fmt"
	"log"
	"net"
	"net/http"
	"os"
	"os/signal"
	"path/filepath"

	_ "github.com/mattn/go-sqlite3"

	"de.heapoverflow/ordertracker"
	"de.heapoverflow/ordertracker/logger"
)

var version = "unknown"

func usage() {
	fmt.Fprintf(os.Stderr, "usage: %s [arguments]\n", os.Args[0])
	flag.PrintDefaults()
}

func main() {
	pwd, err := os.Getwd()
	if err != nil {
		panic(err)
	}
	defaultDataDir := filepath.Join(pwd, "ordertracker-data")

	var showVersion bool
	var debug bool
	var dataDir string
	var logFilename string
	var listenAddrPort string
	flag.BoolVar(&showVersion, "version", false,
		"Show program version and exit")
	flag.BoolVar(&debug, "debug", false,
		"Enable debug level logging")
	flag.StringVar(&dataDir, "data-dir", defaultDataDir,
		"Path for storing data files")
	flag.StringVar(&logFilename, "logfile", "", "Path to logfile")
	flag.StringVar(&listenAddrPort, "listen", "localhost:4141",
		"Address and port to listen on")
	flag.Usage = func() {
		usage()
		os.Exit(0)
	}
	flag.Parse()

	if flag.NArg() != 0 {
		usage()
		os.Exit(2)
	}

	if _, _, err := net.SplitHostPort(listenAddrPort); err != nil {
		fmt.Fprintf(os.Stderr, "invalid address or port: %s\n", err)
		os.Exit(1)
	}

	if showVersion {
		// show version and quit
		fmt.Printf("%s %s\n", os.Args[0], version)
		os.Exit(0)
	}

	// create data directory
	documentDir := filepath.Join(dataDir, "documents")
	err = os.MkdirAll(documentDir, 0750)
	if err != nil && !os.IsExist(err) {
		fmt.Fprintf(os.Stderr,
			"failed to create documents directory: %s\n", err)
		os.Exit(1)
	}

	// set up database
	dbPath := fmt.Sprintf("file:%s/ordertracker.sqlite3", dataDir)
	db, err := sql.Open("sqlite3", dbPath)
	if err != nil {
		fmt.Fprintf(os.Stderr, "failed to open database: %s\n", err)
		os.Exit(1)
	}
	// test database connection
	if err := db.Ping(); err != nil {
		fmt.Fprintf(os.Stderr, "failed to connect to database: %s\n",
			err)
		os.Exit(1)
	}
	defer db.Close()

	// set up logger
	logFile := os.Stderr
	if logFilename != "" {
		logFile, err = os.Create(logFilename)
		if err != nil {
			fmt.Fprintf(os.Stderr, "failed to open logfile: %s\n",
				err)
			os.Exit(1)
		}
	}
	logLevel := logger.LogInfo
	if debug {
		logLevel = logger.LogDebug
	}
	lg := logger.New(logFile, log.LstdFlags, logLevel)
	lg.Infof("ordertracker version %s starting up", version)

	// start ordertracker application
	s, err := ordertracker.NewOrderTrackerServer(lg, db, listenAddrPort,
		documentDir)
	if err != nil {
		fmt.Fprintf(os.Stderr, "failed to instantiate server: %s\n",
			err)
		os.Exit(1)
	}

	// graceful shutdown on SIGINT
	idleConnsClosed := make(chan struct{})
	go func() {
		// signal that shutdown is complete
		defer close(idleConnsClosed)

		sigint := make(chan os.Signal, 1)
		signal.Notify(sigint, os.Interrupt)
		<-sigint
		lg.Info("received SIGINT, shutting down")
		if err := s.Shutdown(); err != nil {
			fmt.Fprintf(os.Stderr,
				"failed to gracefully shut down: %s\n", err)
			os.Exit(1)
		}
	}()

	// listen for connections and handle incoming requests
	lg.Infof("listen on %s", listenAddrPort)
	err = s.Run()
	if err != http.ErrServerClosed {
		// failed to listen or runtime error
		fmt.Fprintf(os.Stderr, "failed to run server: %s\n", err)
		os.Exit(1)
	}

	// wait for graceful shutdown to complete
	<-idleConnsClosed

	lg.Info("ordertracker server stopped")

	if logFilename != "" {
		logFile.Sync()
		logFile.Close()
	}

	os.Exit(1)
}
