//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

package ordertracker

import (
	"hash"
	"io"
	"net/http"
	"net/url"
	"path"
	"strings"
)

type ReadSeekCloser interface {
	io.Reader
	io.Seeker
	io.Closer
}

type HashWriter struct {
	io.Writer
	hash.Hash
}

func NewHashWriter(w io.Writer, h hash.Hash) *HashWriter {
	return &HashWriter{w, h}
}

func (w HashWriter) Write(p []byte) (n int, err error) {
	w.Hash.Write(p)
	return w.Writer.Write(p)
}

const sniffLen = 512

type ContentTypeReader struct {
	io.Reader
	buf []byte
}

func NewContentTypeReader(r io.Reader) *ContentTypeReader {
	return &ContentTypeReader{Reader: r, buf: make([]byte, 0, sniffLen)}
}

func (r *ContentTypeReader) Read(dst []byte) (int, error) {
	n, err := r.Reader.Read(dst)
	if (err == nil || err == io.EOF) && n > 0 && len(r.buf) < cap(r.buf) {
		appendLen := cap(r.buf) - len(r.buf)
		if n < appendLen {
			appendLen = n
		}
		r.buf = append(r.buf, dst[:appendLen]...)
	}
	return n, err
}

func (r *ContentTypeReader) DetectContentType() string {
	return http.DetectContentType(r.buf)
}

func ShiftPath(requestPath string) (string, string) {
	// normalize without trailing slash
	normPath := path.Clean("/" + requestPath)
	i := strings.IndexByte(normPath[1:], '/')
	if i < 0 {
		return normPath, "/"
	}
	return normPath[:i+1], normPath[i+1:]
}

func AppendURLPath(u *url.URL, part string) *url.URL {
	u2 := new(url.URL)
	*u2 = *u
	u2.Path = path.Join(u2.Path, part)
	return u2
}

func ReplacePath(path string, h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		r2 := new(http.Request)
		*r2 = *r
		r2.URL = new(url.URL)
		*r2.URL = *r.URL
		r2.URL.Path = path
		h.ServeHTTP(w, r2)
	})
}

func HTTPErrorHandler(text string, status int) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		http.Error(w, text, status)
	})
}
