//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

package logger

import (
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"sync"
)

const (
	LogNone = iota
	LogWarn
	LogInfo
	LogDebug
)

const (
	tagWarn  = "WARN:  "
	tagInfo  = "INFO:  "
	tagDebug = "DEBUG: "
)

type Logger struct {
	mu       sync.Mutex
	logLevel int
	writer   io.Writer
	warn     *log.Logger
	info     *log.Logger
	debug    *log.Logger
}

func New(out io.Writer, flag, logLevel int) *Logger {
	var warnLogger, infoLogger, debugLogger *log.Logger
	if logLevel >= LogWarn {
		warnLogger = log.New(out, tagWarn, flag)
	} else {
		warnLogger = log.New(ioutil.Discard, tagWarn, flag)
	}
	if logLevel >= LogInfo {
		infoLogger = log.New(out, tagInfo, flag)
	} else {
		infoLogger = log.New(ioutil.Discard, tagInfo, flag)
	}
	if logLevel >= LogDebug {
		debugLogger = log.New(out, tagDebug, flag)
	} else {
		debugLogger = log.New(ioutil.Discard, tagDebug, flag)
	}

	return &Logger{
		logLevel: logLevel,
		writer:   out,
		warn:     warnLogger,
		info:     infoLogger,
		debug:    debugLogger,
	}
}

func (l *Logger) output(logLevel, depth int, s string) {
	l.mu.Lock()
	defer l.mu.Unlock()
	switch logLevel {
	case LogWarn:
		l.warn.Output(depth, s)
	case LogInfo:
		l.info.Output(depth, s)
	case LogDebug:
		l.debug.Output(depth, s)
	default:
		panic("invalid log level")
	}
}

func (l *Logger) WarnLogger() *log.Logger {
	return l.warn
}

func (l *Logger) Warn(v ...interface{}) {
	l.output(LogWarn, 2, fmt.Sprint(v...))
}

func (l *Logger) Warnf(format string, v ...interface{}) {
	l.output(LogWarn, 2, fmt.Sprintf(format, v...))
}

func (l *Logger) Warnln(v ...interface{}) {
	l.output(LogWarn, 2, fmt.Sprintln(v...))
}

func (l *Logger) InfoLogger() *log.Logger {
	return l.info
}

func (l *Logger) Info(v ...interface{}) {
	l.output(LogInfo, 2, fmt.Sprint(v...))
}

func (l *Logger) Infof(format string, v ...interface{}) {
	l.output(LogInfo, 2, fmt.Sprintf(format, v...))
}

func (l *Logger) Infoln(v ...interface{}) {
	l.output(LogInfo, 2, fmt.Sprintln(v...))
}

func (l *Logger) DebugLogger() *log.Logger {
	return l.debug
}

func (l *Logger) Debug(v ...interface{}) {
	l.output(LogDebug, 2, fmt.Sprint(v...))
}

func (l *Logger) Debugf(format string, v ...interface{}) {
	l.output(LogDebug, 2, fmt.Sprintf(format, v...))
}

func (l *Logger) Debugln(v ...interface{}) {
	l.output(LogDebug, 2, fmt.Sprintln(v...))
}
