//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

// +build ignore

package main

import (
	"bufio"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"sort"
	"strings"
)

func generateFileEntry(filePath string, w *bufio.Writer, r *bufio.Reader) error {
	buf := make([]byte, 1024)
	fmt.Fprintf(w, "\t%q: \"", filePath)
	for {
		n, err := r.Read(buf)
		for i := 0; i < n; i++ {
			switch {
			case buf[i] == '"' || buf[i] == '\\':
				w.WriteByte('\\')
				w.WriteByte(buf[i])
			case buf[i] == '\t':
				w.WriteByte('\\')
				w.WriteByte('t')
			case buf[i] == '\n':
				w.WriteByte('\\')
				w.WriteByte('n')
			case buf[i] == '\r':
				w.WriteByte('\\')
				w.WriteByte('r')
			case buf[i] >= 0x20 && buf[i] <= 0x7e:
				w.WriteByte(buf[i])
			default:
				fmt.Fprintf(w, "\\x%02x", buf[i])
			}
		}
		if err == io.EOF {
			break
		} else if err != nil {
			return err
		}
	}
	fmt.Fprintf(w, "\",\n")

	return nil
}

func generateStatic(paths map[string]string, develPaths map[string]string, filePackage string, outFile string) (err error) {
	var tmpfile *os.File
	var tmpname string
	var w *bufio.Writer = bufio.NewWriter(os.Stdout)
	if outFile != "" {
		tmpfile, err = ioutil.TempFile(filepath.Dir(outFile),
			filepath.Base(outFile)+".*")
		if err != nil {
			err = fmt.Errorf("failed to create temporary file: %w",
				err)
			return
		}
		tmpname = tmpfile.Name()
		defer func() {
			if err != nil {
				tmpfile.Close()
				os.Remove(tmpname)
			}
		}()
		w = bufio.NewWriter(tmpfile)
	}

	fmt.Fprintf(w, "// Code generated by \"makestatic\"; DO NOT EDIT.\n\n")
	fmt.Fprintf(w, "package %s\n\n", filePackage)

	// sort keys for reproducible output
	var staticDevelKeys []string
	for k := range develPaths {
		staticDevelKeys = append(staticDevelKeys, k)
	}
	sort.Strings(staticDevelKeys)

	fmt.Fprintf(w, "var StaticDevelPaths = map[string]string{\n")
	for _, destPath := range staticDevelKeys {
		fmt.Fprintf(w, "\t%q: %q,\n", destPath, develPaths[destPath])
	}
	fmt.Fprintf(w, "}\n\n")

	// sort keys for reproducible output
	var staticFilesKeys []string
	for k := range paths {
		staticFilesKeys = append(staticFilesKeys, k)
	}
	sort.Strings(staticFilesKeys)

	fmt.Fprintf(w, "var StaticFiles = map[string]string{\n")
	for _, destPath := range staticFilesKeys {
		var f *os.File
		f, err = os.Open(paths[destPath])
		if err != nil {
			return
		}
		defer f.Close()
		r := bufio.NewReader(f)
		if err = generateFileEntry(destPath, w, r); err != nil {
			return
		}
	}
	fmt.Fprintf(w, "}\n")

	if err = w.Flush(); err != nil {
		err = fmt.Errorf("failed to flush output file: %w", err)
		return
	}
	if outFile != "" {
		if err = tmpfile.Close(); err != nil {
			err = fmt.Errorf("failed to close output file: %w",
				err)
			return
		}
		if err = os.Rename(tmpname, outFile); err != nil {
			err = fmt.Errorf("failed to rename output file: %w",
				err)
			return
		}
	}

	return
}

type stringsFlag []string

func (s *stringsFlag) String() string {
	return fmt.Sprint(*s)
}

func (s *stringsFlag) Set(value string) error {
	*s = append(*s, value)
	return nil
}

func usage() {
	fmt.Fprintf(os.Stderr, "usage: %s [arguments]\n", os.Args[0])
	flag.PrintDefaults()
}

func main() {
	var dirs, mappings stringsFlag
	var filePackage, outputFile string
	flag.Var(&dirs, "dir", "Include a directory")
	flag.Var(&mappings, "map",
		"Include a file and map its path onto another path")
	flag.StringVar(&filePackage, "package", "main", "Package name")
	flag.StringVar(&outputFile, "output", "", "Output filename")
	flag.Usage = func() {
		usage()
		os.Exit(0)
	}
	flag.Parse()

	if flag.NArg() != 0 {
		usage()
		os.Exit(2)
	}

	if filePackage == "" {
		fmt.Fprintf(os.Stderr, "package name must not be empty\n")
		os.Exit(1)
	}

	paths := make(map[string]string)
	develPaths := make(map[string]string)

	for i := range dirs {
		err := filepath.Walk(dirs[i],
			func(path string, info os.FileInfo, err error) error {
				if err != nil {
					return err
				}
				if info.IsDir() {
					return nil
				}

				destPath, err := filepath.Rel(dirs[i], path)
				if err != nil {
					return err
				}
				destPath = filepath.ToSlash(destPath)
				paths[destPath] = path

				return nil
			})
		if err != nil {
			fmt.Fprintf(os.Stderr,
				"failed to traverse directory %s: %s\n",
				dirs[i], err)
			os.Exit(1)
		}
	}

	for i := range mappings {
		n := strings.IndexByte(mappings[i], filepath.ListSeparator)
		if n == -1 {
			fmt.Fprintf(os.Stderr, "invalid mapping: %s\n",
				mappings[i])
			os.Exit(1)
		}

		srcPath := mappings[i][:n]
		destPath := filepath.ToSlash(mappings[i][n+1:])
		if len(srcPath) == 0 || len(destPath) == 0 {
			fmt.Fprintf(os.Stderr, "invalid mapping: %s\n",
				mappings[i])
			os.Exit(1)
		}
		paths[destPath] = srcPath
		develPaths[destPath] = srcPath
	}

	if err := generateStatic(paths, develPaths, filePackage, outputFile); err != nil {
		fmt.Fprintf(os.Stderr, "failed to generate file: %s\n", err)
		os.Exit(1)
	}
}
