//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

//go:generate go run makestatic.go -dir static -map LICENSE:license -package ordertracker -output staticfiles.go

package ordertracker

import (
	"context"
	"crypto/rand"
	"crypto/sha256"
	"database/sql"
	"database/sql/driver"
	"encoding/base64"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"io"
	"mime"
	"net/http"
	"net/url"
	"os"
	"path"
	"path/filepath"
	"strings"
	"time"

	sqlite3 "github.com/mattn/go-sqlite3"
)

const maxUploadSize = 10 * 1024 * 1024

type DocumentType int

const (
	_ DocumentType = iota
	OrderDocument
	DemandPlanDocument
	ConfirmationDocument
)

type NonexistentOrderError struct {
	ID OrderID
}

func (e *NonexistentOrderError) Error() string {
	return fmt.Sprintf("order %s does not exist", e.ID)
}

type OrderExistsError struct {
	ID OrderID
}

func (e *OrderExistsError) Error() string {
	return fmt.Sprintf("order %s already exist", e.ID)
}

type NonexistentDocumentError struct {
	ID   OrderID
	Type DocumentType
}

func (e *NonexistentDocumentError) Error() string {
	switch e.Type {
	case OrderDocument:
		return fmt.Sprintf("order document for order %s does not exist", e.ID)
	case DemandPlanDocument:
		return fmt.Sprintf("demand plan document for order %s does not exist", e.ID)
	case ConfirmationDocument:
		return fmt.Sprintf("confirmation document for order %s does not exist", e.ID)
	default:
		return fmt.Sprintf("unknown document for order %s does not exist", e.ID)
	}
}

type InvalidContentTypeError struct {
	ContentType string
}

func (e *InvalidContentTypeError) Error() string {
	return fmt.Sprintf("invalid content type: %s", e.ContentType)
}

type OrderID [16]byte

func NewOrderID() OrderID {
	var id OrderID

	// 64 bits UNIX time (ns)
	t := time.Now().UnixNano()
	binary.BigEndian.PutUint64(id[:8], uint64(t))

	// 64 bits random data
	if _, err := rand.Read(id[8:]); err != nil {
		panic(err)
	}

	return id
}

func (id OrderID) String() string {
	return base64.RawURLEncoding.EncodeToString(id[:])
}

func (id *OrderID) ParseString(s string) error {
	b, err := base64.RawURLEncoding.DecodeString(s)
	if err != nil {
		return err
	}
	return id.ParseBytes(b)
}

func (id *OrderID) ParseBytes(b []byte) error {
	if len(b) != len(id) {
		return fmt.Errorf("invalid ID length: %d", len(b))
	}
	copy(id[:], b)
	return nil
}

func (id *OrderID) Scan(src interface{}) error {
	b, ok := src.([]byte)
	if !ok {
		return fmt.Errorf("invalid type for an OrderID: %T", src)
	}
	copy(id[:], b)
	return nil
}

func (id OrderID) Value() (driver.Value, error) {
	return driver.Value(id[:]), nil
}

type Order struct {
	ID                   OrderID   `json:"-"`
	OrderNo              string    `json:"order_no"`
	CreatedDate          time.Time `json:"created_date"`
	PreferredDate        time.Time `json:"preferred_date"`
	ConfirmedDate        time.Time `json:"confirmed_date"`
	OrderDocument        string    `json:"-"`
	DemandPlanDocument   string    `json:"-"`
	ConfirmationDocument string    `json:"-"`
	Comment              string    `json:"comment"`
}

type OrderFilter struct {
	OrderNo       string
	CreatedFrom   time.Time
	CreatedTo     time.Time
	PreferredFrom time.Time
	PreferredTo   time.Time
	ConfirmedFrom time.Time
	ConfirmedTo   time.Time
}

type OrderResp struct {
	*Order
	baseURL *url.URL
}

func (r *OrderResp) MarshalJSON() ([]byte, error) {
	selfPath := path.Join(r.baseURL.Path, r.ID.String())
	links := struct {
		Self         string `json:"self"`
		Order        string `json:"order"`
		DemandPlan   string `json:"demand_plan"`
		Confirmation string `json:"confirmation"`
	}{
		Self:         selfPath,
		Order:        path.Join(selfPath, "order"),
		DemandPlan:   path.Join(selfPath, "demand-plan"),
		Confirmation: path.Join(selfPath, "confirmation"),
	}

	type OrderRespAlias OrderResp // avoid infinite loop when marshalling
	response := struct {
		*OrderRespAlias
		OrderExists        bool        `json:"order_exists"`
		DemandPlanExists   bool        `json:"demand_plan_exists"`
		ConfirmationExists bool        `json:"confirmation_exists"`
		Links              interface{} `json:"links"`
	}{
		OrderRespAlias:     (*OrderRespAlias)(r),
		OrderExists:        r.Order.OrderDocument != "",
		DemandPlanExists:   r.Order.DemandPlanDocument != "",
		ConfirmationExists: r.Order.ConfirmationDocument != "",
		Links:              links,
	}

	return json.Marshal(response)
}

type OrderCollectionResp struct {
	orders  []*Order
	baseURL *url.URL
}

func (r *OrderCollectionResp) MarshalJSON() ([]byte, error) {
	ordersResp := make([]*OrderResp, 0, len(r.orders))
	for i, _ := range r.orders {
		ordersResp = append(ordersResp,
			&OrderResp{r.orders[i], r.baseURL})
	}

	links := struct {
		Self string `json:"self"`
	}{
		r.baseURL.EscapedPath(),
	}

	response := struct {
		Orders []*OrderResp `json:"orders"`
		Links  interface{}  `json:"links"`
	}{
		ordersResp,
		links,
	}

	return json.Marshal(response)
}

type OrderManager struct {
	db          *sql.DB
	documentDir string
}

func NewOrderManager(db *sql.DB, documentDir string) *OrderManager {
	return &OrderManager{db: db, documentDir: documentDir}
}

func (m *OrderManager) Add(ctx context.Context, order *Order) error {
	_, err := m.db.ExecContext(ctx, `
		INSERT INTO orders (
			id,
			order_no,
			created_date,
			preferred_date,
			confirmed_date,
			comment,
			order_document,
			demand_plan_document,
			confirmation_document
		)
		VALUES(
			:id,
			trim(:order_no),
			:created_date,
			:preferred_date,
			:confirmed_date,
			:comment,
			'',
			'',
			''
		)`,
		sql.Named("id", order.ID),
		sql.Named("order_no", order.OrderNo),
		sql.Named("created_date", order.CreatedDate),
		sql.Named("preferred_date", order.PreferredDate),
		sql.Named("confirmed_date", order.ConfirmedDate),
		sql.Named("comment", order.Comment))
	if err != nil {
		if driverErr, ok := err.(sqlite3.Error); ok {
			if driverErr.Code == sqlite3.ErrConstraint &&
				driverErr.ExtendedCode == sqlite3.ErrConstraintUnique {
				err = &OrderExistsError{order.ID}
			}
		}
		return err
	}
	return nil
}

func (m *OrderManager) GetAll(ctx context.Context, filter *OrderFilter) ([]*Order, error) {
	conditions := make([]string, 0, 7)
	if filter.OrderNo != "" {
		conditions = append(conditions, `
			(order_no
			LIKE '%' || replace(replace(replace(trim(:order_no),
				'\', '\\'), '%', '\%'), '_', '\_') || '%'
			ESCAPE '\')
			`)
	}
	if !filter.CreatedFrom.IsZero() {
		conditions = append(conditions, `
			datetime(created_date) >= datetime(:created_from)
			`)
	}
	if !filter.CreatedTo.IsZero() {
		conditions = append(conditions, `
			datetime(created_date) <= datetime(:created_to)
			`)
	}
	if !filter.PreferredFrom.IsZero() {
		conditions = append(conditions, `
			preferred_date >= datetime(:preferred_from)
			`)
	}
	if !filter.PreferredTo.IsZero() {
		conditions = append(conditions, `
			preferred_date <= datetime(:preferred_to)
			`)
	}
	if !filter.ConfirmedFrom.IsZero() {
		conditions = append(conditions, `
			confirmed_date >= datetime(:confirmed_from)
			`)
	}
	if !filter.ConfirmedTo.IsZero() {
		conditions = append(conditions, `
			confirmed_date <= datetime(:confirmed_to)
			`)
	}
	var whereClause string
	if len(conditions) > 0 {
		whereClause = "WHERE " + strings.Join(conditions, " AND ")
	}

	rows, err := m.db.QueryContext(ctx, `
		SELECT
			id,
			order_no,
			created_date,
			preferred_date,
			confirmed_date,
			comment,
			order_document,
			demand_plan_document,
			confirmation_document
		FROM orders
		`+whereClause+`
		ORDER BY
			created_date DESC,
			order_no DESC`,
		sql.Named("order_no", filter.OrderNo),
		sql.Named("created_from", filter.CreatedFrom),
		sql.Named("created_to", filter.CreatedTo),
		sql.Named("preferred_from", filter.PreferredFrom),
		sql.Named("preferred_to", filter.PreferredTo),
		sql.Named("confirmed_from", filter.ConfirmedFrom),
		sql.Named("confirmed_to", filter.ConfirmedTo))
	if err != nil {
		return nil, err
	}
	defer rows.Close()
	orders := make([]*Order, 0)

	for rows.Next() {
		order := &Order{}
		if err := rows.Scan(&order.ID, &order.OrderNo,
			&order.CreatedDate, &order.PreferredDate,
			&order.ConfirmedDate, &order.Comment,
			&order.OrderDocument, &order.DemandPlanDocument,
			&order.ConfirmationDocument); err != nil {
			return nil, err
		}
		orders = append(orders, order)
	}

	return orders, nil
}

func (m *OrderManager) Get(ctx context.Context, id OrderID) (*Order, error) {
	order := &Order{}
	err := m.db.QueryRowContext(ctx, `
		SELECT
			id,
			order_no,
			created_date,
			preferred_date,
			confirmed_date,
			comment,
			order_document,
			demand_plan_document,
			confirmation_document
		FROM orders
		WHERE id = ?`,
		id).Scan(&order.ID, &order.OrderNo, &order.CreatedDate,
		&order.PreferredDate, &order.ConfirmedDate, &order.Comment,
		&order.OrderDocument, &order.DemandPlanDocument,
		&order.ConfirmationDocument)
	if err == sql.ErrNoRows {
		return nil, &NonexistentOrderError{id}
	} else if err != nil {
		return nil, err
	}
	return order, nil
}

func (m *OrderManager) Update(ctx context.Context, order *Order) error {
	result, err := m.db.ExecContext(ctx, `
		UPDATE orders
		SET
			comment = :comment,
			created_date = :created_date,
			preferred_date = :preferred_date,
			confirmed_date = :confirmed_date
		WHERE id = :id`,
		sql.Named("id", order.ID),
		sql.Named("comment", order.Comment),
		sql.Named("created_date", order.CreatedDate),
		sql.Named("preferred_date", order.PreferredDate),
		sql.Named("confirmed_date", order.ConfirmedDate))
	if err != nil {
		return err
	}
	if rows, err := result.RowsAffected(); err != nil {
		return err
	} else if rows != 1 {
		return &NonexistentOrderError{order.ID}
	}
	return nil
}

func (m *OrderManager) Delete(ctx context.Context, id OrderID) error {
	result, err := m.db.ExecContext(ctx, `
		DELETE FROM orders
		WHERE id = ?`,
		id)
	if err != nil {
		return err
	}
	if rows, err := result.RowsAffected(); err != nil {
		return err
	} else if rows != 1 {
		return &NonexistentOrderError{id}
	}
	return nil
}

func (m *OrderManager) GetDocument(ctx context.Context, id OrderID, docType DocumentType) (ReadSeekCloser, time.Time, error) {
	order, err := m.Get(ctx, id)
	if err != nil {
		return nil, time.Time{}, err
	}
	var document string
	switch docType {
	case OrderDocument:
		document = order.OrderDocument
	case DemandPlanDocument:
		document = order.DemandPlanDocument
	case ConfirmationDocument:
		document = order.ConfirmationDocument
	}
	if document == "" {
		return nil, time.Time{}, &NonexistentDocumentError{id, docType}
	}
	filename := filepath.Join(m.documentDir, document)

	f, err := os.Open(filename)
	if err != nil {
		return nil, time.Time{},
			fmt.Errorf("failed to open document: %w", err)
	}
	defer func() {
		if err != nil {
			f.Close()
		}
	}()

	// ensure this is a regular file
	d, err := f.Stat()
	if err != nil {
		return nil, time.Time{},
			fmt.Errorf("failed to stat document: %w", err)
	}
	if !d.Mode().IsRegular() {
		return nil, time.Time{}, &NonexistentDocumentError{id, docType}
	}

	return f, d.ModTime(), nil
}

func (m *OrderManager) uploadFile(id OrderID, docType DocumentType, r io.Reader) (docName string, hashSum []byte, err error) {
	var basename string
	switch docType {
	case OrderDocument:
		basename = "order"
	case DemandPlanDocument:
		basename = "demand-plan"
	case ConfirmationDocument:
		basename = "confirmation"
	}

	var filename string
	var f *os.File
	for i := 0; i < 1000; i++ {
		// 64 bits random data
		var b [8]byte
		if _, err = rand.Read(b[:]); err != nil {
			panic(err)
		}
		randomSuffix := base64.RawURLEncoding.EncodeToString(b[:])
		filename = fmt.Sprintf("%s-%s-%s.pdf", basename, id,
			randomSuffix)

		f, err = os.OpenFile(filepath.Join(m.documentDir, filename),
			os.O_WRONLY|os.O_CREATE|os.O_EXCL, 0750)
		if os.IsExist(err) {
			continue
		} else if err != nil {
			return
		}
		break
	}
	defer func() {
		if err != nil {
			os.Remove(docName)
			f.Close()
		}
	}()
	hashWriter := NewHashWriter(f, sha256.New())
	contentTypeReader := NewContentTypeReader(r)
	if _, err = io.Copy(hashWriter, contentTypeReader); err != nil {
		return
	}
	if contentType := contentTypeReader.DetectContentType();
		contentType != "application/pdf" {
		err = &InvalidContentTypeError{contentType}
		return
	}
	if err = f.Sync(); err != nil {
		return
	}
	if err = f.Close(); err != nil {
		return
	}
	docName = filename
	hashSum = hashWriter.Sum(nil)

	return
}

func (m *OrderManager) updateOrderDocument(ctx context.Context, id OrderID, docType DocumentType, docName string, docHashSum []byte) (oldDocName string, err error) {
	var selectDoc string
	var updateDoc string
	switch docType {
	case OrderDocument:
		selectDoc = `
			SELECT
				order_document
			FROM orders
			WHERE id = :id`
		updateDoc = `
			UPDATE orders
			SET
				order_document = :document,
				order_document_hash = :document_hash
			WHERE id = :id`
	case DemandPlanDocument:
		selectDoc = `
			SELECT
				demand_plan_document
			FROM orders
			WHERE id = :id`
		updateDoc = `
			UPDATE orders
			SET
				demand_plan_document = :document,
				demand_plan_document_hash = :document_hash
			WHERE id = :id`
	case ConfirmationDocument:
		selectDoc = `
			SELECT
				confirmation_document
			FROM orders
			WHERE id = :id`
		updateDoc = `
			UPDATE orders
			SET
				confirmation_document = :document,
				confirmation_document_hash = :document_hash
			WHERE id = :id`
	}

	// get the old document name and update it in a single transaction
	tx, err := m.db.BeginTx(ctx,
		&sql.TxOptions{Isolation: sql.LevelSerializable})
	if err != nil {
		return
	}
	defer func() {
		if err != nil {
			tx.Rollback()
			os.Remove(filepath.Join(m.documentDir, docName))
		}
	}()

	var tmpDocName string
	err = tx.QueryRowContext(ctx, selectDoc,
		sql.Named("id", id)).Scan(&tmpDocName)
	if err == sql.ErrNoRows {
		err = &NonexistentOrderError{id}
		return
	} else if err != nil {
		return
	}

	result, err := tx.ExecContext(ctx, updateDoc, sql.Named("id", id),
		sql.Named("document", docName),
		sql.Named("document_hash", docHashSum))
	if err != nil {
		return
	}
	rows, err := result.RowsAffected()
	if err != nil {
		return
	} else if rows != 1 {
		err = &NonexistentOrderError{id}
		return
	}

	if err = tx.Commit(); err != nil {
		return
	}

	oldDocName = tmpDocName

	return
}

func (m *OrderManager) UpdateDocument(ctx context.Context, id OrderID, docType DocumentType, r io.Reader) error {
	// upload the document file, calculate a file hash and get its
	// (randomized) filename in order to store it in the database
	docName, docHashSum, err := m.uploadFile(id, docType, r)
	if err != nil {
		return err
	}

	oldDocName, err := m.updateOrderDocument(ctx, id, docType, docName,
		docHashSum)
	if err != nil {
		return err
	}

	// finally remove the old document file
	if oldDocName != "" {
		os.Remove(filepath.Join(m.documentDir, oldDocName))
	}

	return nil
}

func (m *OrderManager) DeleteDocument(ctx context.Context, id OrderID, docType DocumentType) error {
	order, err := m.Get(ctx, id)
	if err != nil {
		return err
	}
	var document string
	switch docType {
	case OrderDocument:
		document = order.OrderDocument
	case DemandPlanDocument:
		document = order.DemandPlanDocument
	case ConfirmationDocument:
		document = order.ConfirmationDocument
	}
	if document == "" {
		return &NonexistentDocumentError{id, docType}
	}

	// remove reference from database
	var updateDoc string
	switch docType {
	case OrderDocument:
		updateDoc = `
			UPDATE orders
			SET
				order_document = '',
				order_document_hash = ''
			WHERE id = :id`
	case DemandPlanDocument:
		updateDoc = `
			UPDATE orders
			SET
				demand_plan_document = '',
				demand_plan_document_hash = ''
			WHERE id = :id`
	case ConfirmationDocument:
		updateDoc = `
			UPDATE orders
			SET
				confirmation_document = '',
				confirmation_document_hash = ''
			WHERE id = :id`
	}
	result, err := m.db.ExecContext(ctx, updateDoc, sql.Named("id", id))
	if err != nil {
		return err
	}
	if rows, err := result.RowsAffected(); err != nil {
		return err
	} else if rows != 1 {
		return &NonexistentOrderError{id}
	}

	filename := filepath.Join(m.documentDir, document)
	err = os.Remove(filename)
	if err != nil {
		return fmt.Errorf("failed to delete %s: %w", filename, err)
	}

	return nil
}

type OrderHandler struct {
	ot      *OrderTracker
	manager *OrderManager
	baseURL *url.URL
}

func NewOrderHandler(ot *OrderTracker, baseURL *url.URL) *OrderHandler {
	h := &OrderHandler{
		ot:      ot,
		manager: NewOrderManager(ot.DB, ot.DocumentDir),
		baseURL: baseURL,
	}
	return h
}

func (h *OrderHandler) parseOrderRequest(r *http.Request, id OrderID) (*Order, error) {
	order := &Order{ID: id}
	if err := json.NewDecoder(r.Body).Decode(&order); err != nil {
		return nil, err
	}
	return order, nil
}

func (h *OrderHandler) addOrder() http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// check Content-Type header
		contentTypeHdr := r.Header.Get("Content-Type")
		if contentTypeHdr == "" {
			h.ot.Logger.RequestDebugf(r, "missing content type")
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		mediaType, _, err := mime.ParseMediaType(contentTypeHdr)
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse media type: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		} else if mediaType != "application/json" {
			h.ot.Logger.RequestDebugf(r, "invalid content type: %s",
				mediaType)
			http.Error(w,
				http.StatusText(http.StatusUnsupportedMediaType),
				http.StatusUnsupportedMediaType)
			return
		}

		order, err := h.parseOrderRequest(r, NewOrderID())
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse new order data: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}

		err = h.manager.Add(r.Context(), order)
		if errExists, ok := err.(*OrderExistsError); ok {
			h.ot.Logger.RequestDebugf(r,
				"failed to create order: %s", errExists)
			http.Error(w, http.StatusText(http.StatusConflict),
				http.StatusConflict)
			return
		} else if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to create order %s: %s", order.ID,
				err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}

		h.ot.Logger.RequestDebugf(r, "created order %s", order.ID)
		location := path.Join(h.baseURL.Path, order.ID.String())
		http.Redirect(w, r, location, http.StatusCreated)
	})
}

func parseTimeValue(r *http.Request, name string) (time.Time, error) {
	if value := r.FormValue(name); value != "" {
		return time.Parse(time.RFC3339, value)
	}
	return time.Time{}, nil
}

func (h *OrderHandler) getAllOrders() http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var err error
		filter := &OrderFilter{
			OrderNo: r.FormValue("order_no"),
		}
		filter.CreatedFrom, err = parseTimeValue(r, "created_from")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse created_from value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		filter.CreatedTo, err = parseTimeValue(r, "created_to")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse created_to value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		filter.PreferredFrom, err = parseTimeValue(r, "preferred_from")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse preferred_from value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		filter.PreferredTo, err = parseTimeValue(r, "preferred_to")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse preferred_to value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		filter.ConfirmedFrom, err = parseTimeValue(r, "confirmed_from")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse confirmed_from value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		filter.ConfirmedTo, err = parseTimeValue(r, "confirmed_to")
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse confirmed_to value: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}

		orders, err := h.manager.GetAll(r.Context(), filter)
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to get orders: %s", err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
		response := &OrderCollectionResp{
			orders:  orders,
			baseURL: h.baseURL,
		}

		h.ot.Logger.RequestDebugf(r, "returning all orders")
		w.Header().Add("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(&response); err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to encode orders: %s", err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
	})
}

func (h *OrderHandler) getOrder(id OrderID) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		order, err := h.manager.Get(r.Context(), id)
		if _, ok := err.(*NonexistentOrderError); ok {
			h.ot.Logger.RequestDebugf(r,
				"order %s does not exist", id)
			http.NotFound(w, r)
			return
		} else if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to get order: %s", err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
		response := &OrderResp{order, h.baseURL}

		h.ot.Logger.RequestDebugf(r, "returning order %s", id)
		w.Header().Add("Content-Type", "application/json")
		if err := json.NewEncoder(w).Encode(&response); err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to encode order %s: %s", id, err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
	})
}

func (h *OrderHandler) updateOrder(id OrderID) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		contentTypeHdr := r.Header.Get("Content-Type")
		if contentTypeHdr == "" {
			h.ot.Logger.RequestDebugf(r, "missing content type")
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		mediaType, _, err := mime.ParseMediaType(contentTypeHdr)
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse media type: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		} else if mediaType != "application/json" {
			h.ot.Logger.RequestDebugf(r, "invalid content type: %s",
				mediaType)
			http.Error(w,
				http.StatusText(http.StatusUnsupportedMediaType),
				http.StatusUnsupportedMediaType)
			return
		}

		order, err := h.parseOrderRequest(r, id)
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse order data: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}

		if err := h.manager.Update(r.Context(), order); err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to update order %s: %s", id, err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}

		h.ot.Logger.RequestDebugf(r, "updated order %s", id)
		w.WriteHeader(http.StatusNoContent)
	})
}

func (h *OrderHandler) deleteOrder(id OrderID) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		err := h.manager.Delete(r.Context(), id)
		if _, ok := err.(*NonexistentOrderError); ok {
			h.ot.Logger.RequestDebugf(r, "order %s does not exist",
				id)
			http.NotFound(w, r)
			return
		} else if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to delete order %s: %s", id, err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}

		h.ot.Logger.RequestDebugf(r, "deleted order %s", id)
		w.WriteHeader(http.StatusNoContent)
	})
}

func (h *OrderHandler) uploadDocument(id OrderID, docType DocumentType) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		// check Content-Type header
		contentTypeHdr := r.Header.Get("Content-Type")
		if contentTypeHdr == "" {
			h.ot.Logger.RequestDebugf(r, "missing content type")
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		}
		mediaType, _, err := mime.ParseMediaType(contentTypeHdr)
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to parse media type: %s", err)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		} else if mediaType != "application/pdf" {
			h.ot.Logger.RequestDebugf(r, "invalid content type: %s",
				mediaType)
			http.Error(w, http.StatusText(http.StatusUnsupportedMediaType),
				http.StatusUnsupportedMediaType)
			return
		}

		// check content length against size limits
		if r.ContentLength < 1 {
			h.ot.Logger.RequestDebugf(r,
				"content length invalid: %d bytes",
				r.ContentLength)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		} else if r.ContentLength > maxUploadSize {
			h.ot.Logger.RequestDebugf(r,
				"content length too big: %d bytes",
				r.ContentLength)
			http.Error(w, http.StatusText(http.StatusForbidden),
				http.StatusForbidden)
			return
		}

		// upload, enforce size limit
		err = h.manager.UpdateDocument(r.Context(), id, docType,
			http.MaxBytesReader(w, r.Body, maxUploadSize))
		switch nerr := err.(type) {
		case *NonexistentOrderError:
			h.ot.Logger.RequestDebugf(r, "%s", nerr)
			http.NotFound(w, r)
			return
		case *InvalidContentTypeError:
			h.ot.Logger.RequestDebugf(r, "%s", nerr)
			http.Error(w, http.StatusText(http.StatusBadRequest),
				http.StatusBadRequest)
			return
		case nil:
		default:
			h.ot.Logger.RequestDebugf(r,
				"failed to update document: %s", err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}

		h.ot.Logger.RequestDebugf(r, "uploaded document for order %s",
			id)
		w.WriteHeader(http.StatusNoContent)
	})
}

func (h *OrderHandler) getDocument(id OrderID, docType DocumentType) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		f, modTime, err := h.manager.GetDocument(r.Context(), id,
			docType)
		if _, ok := err.(*NonexistentOrderError); ok {
			h.ot.Logger.RequestDebugf(r, "%s", err)
			http.NotFound(w, r)
			return
		} else if _, ok := err.(*NonexistentDocumentError); ok {
			h.ot.Logger.RequestDebugf(r, "%s", err)
			http.NotFound(w, r)
			return
		} else if err != nil {
			h.ot.Logger.RequestDebugf(r, "%s", err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
		defer f.Close()

		switch docType {
		case OrderDocument:
			h.ot.Logger.RequestDebugf(r,
				"returning order document for order %s", id)
		case DemandPlanDocument:
			h.ot.Logger.RequestDebugf(r,
				"returning demand plan document for order %s", id)
		case ConfirmationDocument:
			h.ot.Logger.RequestDebugf(r,
				"returning confirmation document for order %s", id)
		}
		w.Header().Add("Content-Type", "application/pdf")
		http.ServeContent(w, r, "", modTime, f)
	})
}

func (h *OrderHandler) deleteDocument(id OrderID, docType DocumentType) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		err := h.manager.DeleteDocument(r.Context(), id, docType)
		if _, ok := err.(*NonexistentDocumentError); ok {
			h.ot.Logger.RequestDebugf(r, "%s", err)
			http.NotFound(w, r)
			return
		} else if _, ok := err.(*NonexistentOrderError); ok {
			h.ot.Logger.RequestDebugf(r, "%s", err)
			http.NotFound(w, r)
			return
		} else if err != nil {
			switch docType {
			case OrderDocument:
				h.ot.Logger.RequestDebugf(r,
					"failed to delete order document for order %s: %s", id, err)
			case DemandPlanDocument:
				h.ot.Logger.RequestDebugf(r,
					"failed to delete demand plan document for order %s: %s", id, err)
			case ConfirmationDocument:
				h.ot.Logger.RequestDebugf(r,
					"failed to delete confirmation document for order %s: %s", id, err)
			}
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}

		w.WriteHeader(http.StatusNoContent)
	})
}

func (h *OrderHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	var handler http.Handler
	prefix, remaining := ShiftPath(r.URL.Path)
	if prefix == "/" {
		switch r.Method {
		case "POST":
			handler = h.addOrder()
		case "GET":
			handler = h.getAllOrders()
		default:
			handler = HTTPErrorHandler(http.StatusText(http.StatusMethodNotAllowed),
				http.StatusMethodNotAllowed)
		}

		handler.ServeHTTP(w, r)
		return
	}

	id := OrderID{}
	if err := id.ParseString(prefix[1:]); err != nil {
		h.ot.Logger.RequestDebugf(r, "failed to parse order ID: %s",
			err)
		http.NotFound(w, r)
		return
	}

	prefix, remaining = ShiftPath(remaining)
	switch prefix {
	case "/":
		switch r.Method {
		case "GET":
			handler = h.getOrder(id)
		case "PUT":
			handler = h.updateOrder(id)
		case "DELETE":
			handler = h.deleteOrder(id)
		default:
			handler = HTTPErrorHandler(http.StatusText(http.StatusMethodNotAllowed),
				http.StatusMethodNotAllowed)
		}
	case "/order":
		switch r.Method {
		case "GET":
			handler = h.getDocument(id, OrderDocument)
		case "PUT":
			handler = h.uploadDocument(id, OrderDocument)
		case "DELETE":
			handler = h.deleteDocument(id, OrderDocument)
		default:
			handler = HTTPErrorHandler(http.StatusText(http.StatusMethodNotAllowed),
				http.StatusMethodNotAllowed)
		}
	case "/demand-plan":
		switch r.Method {
		case "GET":
			handler = h.getDocument(id, DemandPlanDocument)
		case "PUT":
			handler = h.uploadDocument(id, DemandPlanDocument)
		case "DELETE":
			handler = h.deleteDocument(id, DemandPlanDocument)
		default:
			handler = HTTPErrorHandler(http.StatusText(http.StatusMethodNotAllowed),
				http.StatusMethodNotAllowed)
		}
	case "/confirmation":
		switch r.Method {
		case "GET":
			handler = h.getDocument(id, ConfirmationDocument)
		case "PUT":
			handler = h.uploadDocument(id, ConfirmationDocument)
		case "DELETE":
			handler = h.deleteDocument(id, ConfirmationDocument)
		default:
			handler = HTTPErrorHandler(http.StatusText(http.StatusMethodNotAllowed),
				http.StatusMethodNotAllowed)
		}
	default:
		handler = http.NotFoundHandler()
	}

	handler.ServeHTTP(w, r)
}
