//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

package ordertracker

import (
	"context"
	"crypto/rand"
	"fmt"
	"io"
	"net"
	"net/http"
	"strings"
	"sync"
	"time"

	"de.heapoverflow/ordertracker/logger"
)

func nextRequestID() string {
	var b [8]byte
	rand.Read(b[:])
	return fmt.Sprintf("%x", b)
}

type ctxKey string

func RequestTracing(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		requestID := r.Header.Get("X-Request-Id")
		if requestID == "" {
			requestID = nextRequestID()
		}
		ctx := context.WithValue(r.Context(), ctxKey("requestID"),
			requestID)
		w.Header().Set("X-Request-Id", requestID)
		h.ServeHTTP(w, r.WithContext(ctx))
	})
}

type tracingResponseWriter struct {
	http.ResponseWriter
	wrapped     http.ResponseWriter
	start       time.Time
	lengthMutex sync.Mutex
	length      int
	statusOnce  sync.Once
	status      int
}

// wrappedResponseWriter() is based on generated code from the httpsnoop
// project (https://github.com/felixge/httpsnoop) and made available under the
// following license terms:
//
// Copyright (c) 2016 Felix Geisendörfer (felix@debuggable.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
func (w *tracingResponseWriter) wrapResponseWriter() http.ResponseWriter {
	f, i0 := w.ResponseWriter.(http.Flusher)
	cn, i1 := w.ResponseWriter.(http.CloseNotifier)
	h, i2 := w.ResponseWriter.(http.Hijacker)
	rf, i3 := w.ResponseWriter.(io.ReaderFrom)
	p, i4 := w.ResponseWriter.(http.Pusher)
	switch {
	// combination 1/32
	case !i0 && !i1 && !i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
		}{w}
	// combination 2/32
	case !i0 && !i1 && !i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Pusher
		}{w, p}
	// combination 3/32
	case !i0 && !i1 && !i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			io.ReaderFrom
		}{w, rf}
	// combination 4/32
	case !i0 && !i1 && !i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			io.ReaderFrom
			http.Pusher
		}{w, rf, p}
	// combination 5/32
	case !i0 && !i1 && i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Hijacker
		}{w, h}
	// combination 6/32
	case !i0 && !i1 && i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Hijacker
			http.Pusher
		}{w, h, p}
	// combination 7/32
	case !i0 && !i1 && i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Hijacker
			io.ReaderFrom
		}{w, h, rf}
	// combination 8/32
	case !i0 && !i1 && i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.Hijacker
			io.ReaderFrom
			http.Pusher
		}{w, h, rf, p}
	// combination 9/32
	case !i0 && i1 && !i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
		}{w, cn}
	// combination 10/32
	case !i0 && i1 && !i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			http.Pusher
		}{w, cn, p}
	// combination 11/32
	case !i0 && i1 && !i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			io.ReaderFrom
		}{w, cn, rf}
	// combination 12/32
	case !i0 && i1 && !i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			io.ReaderFrom
			http.Pusher
		}{w, cn, rf, p}
	// combination 13/32
	case !i0 && i1 && i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			http.Hijacker
		}{w, cn, h}
	// combination 14/32
	case !i0 && i1 && i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			http.Hijacker
			http.Pusher
		}{w, cn, h, p}
	// combination 15/32
	case !i0 && i1 && i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			http.Hijacker
			io.ReaderFrom
		}{w, cn, h, rf}
	// combination 16/32
	case !i0 && i1 && i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.CloseNotifier
			http.Hijacker
			io.ReaderFrom
			http.Pusher
		}{w, cn, h, rf, p}
	// combinartion 17/32
	case i0 && !i1 && !i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
		}{w, f}
	// combination 18/32
	case i0 && !i1 && !i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.Pusher
		}{w, f, p}
	// combination 19/32
	case i0 && !i1 && !i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			io.ReaderFrom
		}{w, f, rf}
	// combination 20/32
	case i0 && !i1 && !i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			io.ReaderFrom
			http.Pusher
		}{w, f, rf, p}
	// combination 21/32
	case i0 && !i1 && i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.Hijacker
		}{w, f, h}
	// combination 22/32
	case i0 && !i1 && i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.Hijacker
			http.Pusher
		}{w, f, h, p}
	// combination 23/32
	case i0 && !i1 && i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.Hijacker
			io.ReaderFrom
		}{w, f, h, rf}
	// combination 24/32
	case i0 && !i1 && i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.Hijacker
			io.ReaderFrom
			http.Pusher
		}{w, f, h, rf, p}
	// combination 25/32
	case i0 && i1 && !i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
		}{w, f, cn}
	// combination 26/32
	case i0 && i1 && !i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			http.Pusher
		}{w, f, cn, p}
	// combination 27/32
	case i0 && i1 && !i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			io.ReaderFrom
		}{w, f, cn, rf}
	// combination 28/32
	case i0 && i1 && !i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			io.ReaderFrom
			http.Pusher
		}{w, f, cn, rf, p}
	// combination 29/32
	case i0 && i1 && i2 && !i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			http.Hijacker
		}{w, f, cn, h}
	// combination 30/32
	case i0 && i1 && i2 && !i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			http.Hijacker
			http.Pusher
		}{w, f, cn, h, p}
	// combination 31/32
	case i0 && i1 && i2 && i3 && !i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			http.Hijacker
			io.ReaderFrom
		}{w, f, cn, h, rf}
	// combination 32/32
	case i0 && i1 && i2 && i3 && i4:
		return struct {
			http.ResponseWriter
			http.Flusher
			http.CloseNotifier
			http.Hijacker
			io.ReaderFrom
			http.Pusher
		}{w, f, cn, h, rf, p}
	}
	panic("unreachable")
}

func (w *tracingResponseWriter) WriteHeader(status int) {
	w.ResponseWriter.WriteHeader(status)

	w.statusOnce.Do(func() { w.status = status })
}

func (w *tracingResponseWriter) Write(b []byte) (int, error) {
	n, err := w.ResponseWriter.Write(b)

	w.lengthMutex.Lock()
	defer w.lengthMutex.Unlock()
	w.length += n

	return n, err
}

func quoteLogField(s string) string {
	s = strings.ReplaceAll(s, `\`, `\\`)
	s = strings.ReplaceAll(s, `"`, `\"`)
	return `"` + s + `"`
}

func formatCLF(w *tracingResponseWriter, r *http.Request) string {
	const timeFmt string = "[02/Jan/2006:15:04:05 -0700]"
	username, _, ok := r.BasicAuth()
	if !ok {
		username = "-"
	}
	host, _, _ := net.SplitHostPort(r.RemoteAddr)
	return fmt.Sprintf("%s %s %s %s \"%s %s %s\" %d %d %s %s",
		host, "-", username, w.start.Format(timeFmt), r.Method,
		r.URL.Path, r.Proto, w.status, w.length,
		quoteLogField(r.Referer()), quoteLogField(r.UserAgent()))
}

type RequestLogger struct {
	*logger.Logger
}

func (b *RequestLogger) getRequestID(r *http.Request) string {
	if v := r.Context().Value(ctxKey("requestID")); v != nil {
		return v.(string)
	}
	return ""
}

func (b *RequestLogger) RequestInfof(r *http.Request, format string, v ...interface{}) {
	b.Infof("[%s] %s", b.getRequestID(r), fmt.Sprintf(format, v...))
}

func (b *RequestLogger) RequestDebugf(r *http.Request, format string, v ...interface{}) {
	b.Debugf("[%s] %s", b.getRequestID(r), fmt.Sprintf(format, v...))
}

func RequestLogging(l *RequestLogger, h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		tw := &tracingResponseWriter{
			ResponseWriter: w,
			start:          time.Now(),
			status:         http.StatusOK,
		}
		l.RequestDebugf(r, "request started: %s %s %s %s", r.RemoteAddr,
			r.Method, r.URL.Path, r.Proto)

		h.ServeHTTP(tw.wrapResponseWriter(), r)

		l.RequestInfof(r, "%s",
			formatCLF(tw, r))
		l.RequestDebugf(r, "request finished in %v",
			time.Since(tw.start))
	})
}
