//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

'use strict';

class UnexpectedResponseError extends Error {
    constructor(status, statusText, ...params) {
        super(...params);

        this.status = status;
        this.statusText = statusText;
        this.message = `Unexpected server response: ${this.status} ${this.statusText}.`;

        if (Error.captureStackTrace) {
            Error.captureStackTrace(this, UnexpectedResponseError);
        }
    }
}

class NonexistentOrderError extends Error {
    constructor(orderURL, ...params) {
        super(...params)

        this.orderURL = orderURL;

        if (Error.captureStackTrace) {
            Error.captureStackTrace(this, NonexistentOrderError);
        }
    }

    get message() {
        return `Order ${this.orderURL} does not exist.`;
    }
}

class OrderExistsError extends Error {
    constructor(orderNo, ...params) {
        super(...params)

        this.orderNo = orderNo;

        if (Error.captureStackTrace) {
            Error.captureStackTrace(this, OrderExistsError);
        }
    }

    get message() {
        return `Order no. ${this.orderNo} already exists.`;
    }
}

class Order {
    constructor(o) {
        this.order_no = o.order_no;
        this.created_date = new Date(o.created_date);
        this.preferred_date = new Date(o.preferred_date);
        this.confirmed_date = new Date(o.confirmed_date);
        this.comment = o.comment;
        this.order_exists =
            (typeof o.order_exists !== 'undefined') ?
            o.order_exists : false;
        this.demand_plan_exists =
            (typeof o.demand_plan_exists !== 'undefined') ?
            o.demand_plan_exists : false;
        this.confirmation_exists =
            (typeof o.confirmation_exists !== 'undefined') ?
            o.confirmation_exists : false;
        this.links = {
            self: '',
            order: '',
            demand_plan: '',
            confirmation: ''
        };
        if (typeof o.links !== 'undefined') {
            this.links.self = o.links.self;
            this.links.order = o.links.order;
            this.links.demand_plan = o.links.demand_plan;
            this.links.confirmation = o.links.confirmation;
        }
    }
}

class OrderCollection {
    constructor(baseURL) {
        this.baseURL = new URL(baseURL);
        this.orders = new Map();
        this.filterData = new FormData();
        this.listeners = {
            refresh: []
        };
    }

    addEventListener(eventType, eventHandler) {
        if (typeof this.listeners[eventType] === 'undefined') {
            return;
        }
        if (this.listeners[eventType].includes(eventHandler)) {
            return;
        }

        this.listeners[eventType].push(eventHandler)
    }

    // private
    emitEvent(e) {
        if (typeof this.listeners[e.type] === 'undefined') {
            return;
        }
        for (let eventHandler of this.listeners[e.type]) {
            eventHandler(e);
        }
    }

    get(orderURL) {
        return this.orders.get(orderURL);
    }

    async refresh() {
        let searchParams = new URLSearchParams(this.filterData);
        let url = new URL(`${this.baseURL}?${searchParams}`);
        let response = await fetch(url);
        if (response.status !== 200) {
            console.log(`failed to get orders: ${response.status} ${response.statusText}`);
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }

        let collection;
        try {
            collection = await response.json();
        } catch(e) {
            console.log('failed to parse JSON response:', e);
            throw(e);
        }
        this.orders.clear();
        for (let order of collection.orders) {
            this.orders.set(order.links.self, new Order(order));
        }

        this.emitEvent({
            type: 'refresh',
            orders: Array.from(this.orders.values())
        });
    }

    async filter(filterData) {
        this.filterData = filterData;
        await this.refresh();
    }

    // private
    async fetchOrder(orderURL) {
        let response = await fetch(orderURL);
        if (response.status === 404) {
            throw(new NonexistentOrderError(orderURL));
        } else if (response.status !== 200) {
            console.log(`failed to get order ${orderURL}: ${response.status} ${response.statusText}`);
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }

        let order;
        try {
            order = await response.json();
        } catch(e) {
            console.log('failed to parse JSON response:', e);
            throw(e);
        }
        // FIXME validate?
        return order;
    }

    async createOrder(order, orderFile, demandPlanFile, confirmationFile) {
        let response = await fetch(this.baseURL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(order)
        });
        if (response.status === 409) {
            throw(new OrderExistsError(order.order_no));
        } else if (response.status != 201) {
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }

        let orderURL = response.headers.get('Location');
        console.log(`created order ${orderURL}`);

        let createdOrder = await this.fetchOrder(orderURL);
        if (typeof orderFile !== 'undefined' && orderFile !== '') {
            await this.updateOrderDocument(createdOrder, orderFile);
        }
        if (typeof demandPlanFile !== 'undefined' && demandPlanFile !== '') {
            await this.updateDemandPlanDocument(createdOrder, demandPlanFile);
        }
        if (typeof confirmationFile !== 'undefined' &&
            confirmationFile !== '') {
            await this.updateConfirmationDocument(createdOrder,
                confirmationFile);
        }

        await this.refresh();
    }

    async updateOrder(orderURL, order, demandPlanFile, confirmationFile) {
        let response = await fetch(orderURL, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(order)
        });
        if (response.status === 404) {
            throw(new NonexistentOrderError(order.links.self));
        } else if (response.status !== 204 && response.status !== 200) {
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }

        console.log(`updated order ${order.links.self}`);

        if (typeof demandPlanFile !== 'undefined' &&
            demandPlanFile !== '') {
            await this.updateDemandPlanDocument(order, demandPlanFile);
        }

        if (typeof confirmationFile !== 'undefined' &&
            confirmationFile !== '') {
            await this.updateConfirmationDocument(order, confirmationFile);
        }

        await this.refresh();
    }

    async deleteOrder(orderURL) {
        let response = await fetch(orderURL, {
            method: 'DELETE'
        })
        if (response.status !== 204 && response.status !== 404) {
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }

        console.log(`deleted order ${orderURL}`);

        await this.refresh();
    }

    // private
    async updateDocument(orderURL, documentURL, documentFile) {
        let response = await fetch(documentURL, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/pdf'
            },
            body: documentFile
        });
        if (response.status === 404) {
            throw(new NonexistentOrderError(orderURL));
        } else if (response.status !== 204) {
            throw(new UnexpectedResponseError(response.status,
                response.statusText));
        }
    }

    async updateOrderDocument(order, orderFile) {
       await this.updateDocument(order.links.self, order.links.order,
           orderFile);
    }

    async updateDemandPlanDocument(order, demandPlanFile) {
        await this.updateDocument(order.links.self, order.links.demand_plan,
            demandPlanFile);
    }

    async updateConfirmationDocument(order, confirmationFile) {
        await this.updateDocument(order.links.self, order.links.confirmation,
            confirmationFile);
    }
}

function getISODateString(date) {
    return date.toISOString().split('T')[0];
}

function setTimeElementToDate(timeElement, s) {
    let date = new Date(s);
    timeElement.dateTime = getISODateString(date);
    document.l10n.setAttributes(timeElement, 'date-value',
        {date: date.valueOf(),});
}

function registerFileDropArea(dropArea) {
    let fileInput = document.getElementById(dropArea.dataset.dropFor);

    for (let eventType of ['dragenter', 'dragover']) {
        dropArea.addEventListener(eventType, e => {
            e.preventDefault();

            dropArea.classList.add('droppable');
        });
    }

    for (let eventType of ['dragleave', 'drop']) {
        dropArea.addEventListener(eventType, e => {
            e.preventDefault();

            dropArea.classList.remove('droppable');
        });
    }

    dropArea.addEventListener('drop', e => {
        let dataTransfer = e.dataTransfer
        let files = dataTransfer.files
        if (files.length !== 1) {
            return;
        }

        // accept file types allowed by the associated input element
        let accept = fileInput.accept;
        let isAcceptable = (accept === '');
        if (!isAcceptable) {
            for (let type of accept.split(',')) {
                if (files[0].type === type) {
                    isAcceptable = true;
                    break;
                }
            }
        }
        if (!isAcceptable) {
            return;
        }

        fileInput.files = files;
    });
}

function createOrderElement(order) {
    let template = document.querySelector('#order-template');
    let clone = template.content.cloneNode(true);

    let orderElement = clone.querySelector('.order');
    orderElement.dataset.href = order.links.self;

    if (order.order_exists) {
        let aElement = document.createElement('a');
        aElement.href = order.links.order;
        aElement.textContent = order.order_no
        aElement.classList.add('view-action')
        aElement.classList.add('order')
        document.l10n.setAttributes(aElement, 'order-link');
        orderElement.querySelector('.order-no').appendChild(aElement);
    } else {
        orderElement.querySelector('.order-no').textContent = order.order_no;
    }

    setTimeElementToDate(orderElement.querySelector('.created-date > time'),
        order.created_date);

    setTimeElementToDate(orderElement.querySelector('.preferred-date > time'),
        order.preferred_date);

    if (order.demand_plan_exists) {
        let aElement = document.createElement('a');
        aElement.href = order.links.demand_plan
        aElement.classList.add('view-action')
        aElement.classList.add('demand-plan')
        document.l10n.setAttributes(aElement, 'demand-plan-link');
        orderElement.querySelector('.demand-plan').appendChild(aElement);
    } else {
        document.l10n.setAttributes(orderElement.querySelector('.demand-plan'),
            'not-available');
    }

    if (order.confirmation_exists) {
        let aElement = document.createElement('a');
        aElement.href = order.links.confirmation
        aElement.classList.add('view-action')
        aElement.classList.add('confirmation')
        document.l10n.setAttributes(aElement, 'confirmation-link');
        orderElement.querySelector('.confirmation').appendChild(aElement);
    } else {
        document.l10n.setAttributes(orderElement.querySelector('.confirmation'),
            'not-confirmed');
    }

    setTimeElementToDate(orderElement.querySelector('.confirmed-date > time'),
        order.confirmed_date);

    orderElement.querySelector('.comment').textContent = order.comment;

    return clone;
}

function initOrdersView(collection) {
    document.querySelector('#orders-actions').addEventListener('click', e => {
        if (e.currentTarget.querySelector('button[name="new"]')
            .contains(e.target)) {
            showNewDialog();
        }
    });

    function clearFilter() {
        collection.filter(new FormData());
        document.querySelector('#filter-info').classList
            .remove('active-filter')
    }

    document.forms['filters'].addEventListener('submit', e => {
        e.preventDefault();
        e.target.blur();

        let orderNo = e.target['order_no'].value.trim();
        let filterDateType = e.target['filter_date_type'].value;
        let fromDate = e.target['from'].valueAsDate;
        let toDate = e.target['to'].valueAsDate;

        let filterData = new FormData()
        if (orderNo !== '') {
            filterData.append('order_no', orderNo)
        }
        if (fromDate !== null) {
            filterData.append(`${filterDateType}_from`, fromDate.toISOString());
        }
        if (toDate !== null) {
            filterData.append(`${filterDateType}_to`, toDate.toISOString())
        }
        // clear filter if there is no input
        if (filterData.entries().next().done) {
            clearFilter();
            return;
        }

        collection.filter(filterData);

        let dateLabel =
            e.target['filter_date_type'].selectedOptions[0].textContent;
        let filterInfoValue = document.querySelector('#filter-info-value');
        document.l10n.setAttributes(filterInfoValue, 'filter-info-value', {
            orderNo: orderNo !== '' ? orderNo : '*',
            dateLabel,
            fromDate: fromDate !== null ? fromDate.valueOf() : 0,
            fromDateISO: fromDate !== null ? fromDate.toISOString() :
                new Date(0).toISOString(),
            toDate: toDate !== null ? toDate.valueOf() : Date.now(),
            toDateISO: toDate !== null ? toDate.toISOString() :
                new Date(0).toISOString()
        });
        document.querySelector('#filter-info').classList.add('active-filter');
    });


    document.forms['filters'].addEventListener('reset', e => {
        e.target.blur();

        clearFilter();
    });

    document.querySelector('#orders-table').addEventListener('click', e => {
        if (e.target.name === 'edit') {
            let orderURL = e.target.closest('tr.order').dataset.href;
            let order = collection.get(orderURL);
            showEditDialog(order);
        } else if (e.target.name === 'delete') {
            let orderURL = e.target.closest('tr.order').dataset.href;
            let order = collection.get(orderURL);
            showDeleteDialog(order);
        } else if (e.target.classList.contains('view-action')) {
            e.preventDefault();

            let orderURL = e.target.closest('tr.order').dataset.href;
            let order = collection.get(orderURL);
            for (let className of e.target.classList) {
                let documentType = className.replace(/-/g, '_');
                if (documentType === 'order' ||
                    documentType === 'demand_plan' ||
                    documentType === 'confirmation') {
                    showViewDialog(order, documentType, e.target.title);
                    break;
                }
            }
        }
    });
}

function refreshOrdersView(orders) {
    let tableBody = document.querySelector('#orders-table > tbody');
    while (tableBody.firstChild) {
        tableBody.removeChild(tableBody.firstChild);
    }

    for (let order of orders) {
        let orderNode = createOrderElement(order);
        document.querySelector('#orders-table > tbody').appendChild(orderNode);
    }
}

function initErrorDialog() {
    let errorDialog = document.querySelector('#error-dialog');
    dialogPolyfill.registerDialog(errorDialog);
}

function showErrorDialog(message, details) {
    let errorDialog = document.querySelector('#error-dialog');
    errorDialog.querySelector('#error-message').textContent = message;
    errorDialog.querySelector('#error-details').textContent = details;
    errorDialog.showModal();
}

function initBusyDialog() {
    let busyDialog = document.querySelector('#busy-dialog');
    dialogPolyfill.registerDialog(busyDialog);
}

function showBusyDialog(message) {
    let busyDialog = document.querySelector('#busy-dialog');
    busyDialog.querySelector('#busy-message').textContent = message;
    busyDialog.showModal();
}

function closeBusyDialog() {
    let busyDialog = document.querySelector('#busy-dialog');
    if (busyDialog.open) {
        busyDialog.close();
    }
}

function initNewDialog(collection) {
    let newDialog = document.querySelector('#new-dialog');
    dialogPolyfill.registerDialog(newDialog);

    for (let dropArea of newDialog.querySelectorAll('.drop-area')) {
        registerFileDropArea(dropArea);
    }

    newDialog.addEventListener('click', e => {
        if (e.target.type === 'reset') {
            newDialog.close('cancel');
        }
    });

    newDialog.addEventListener('close', async e => {
        if (newDialog.returnValue === 'cancel') {
            return;
        }

        let newForm = document.forms['new-order'];
        let formData = new FormData(newForm);
        let order = new Order(Object.fromEntries(formData.entries()));
        let orderDocument = newForm['order_document'].files[0];
        let demandPlanDocument = newForm['demand_plan_document'].files[0];
        let confirmationDocument = newForm['confirmation_document'].files[0];
        showBusyDialog(await document.l10n.formatValue('creating-new-order-message'));
        try {
            await collection.createOrder(order, orderDocument,
                demandPlanDocument, confirmationDocument);
        } catch (e) {
            if (e instanceof OrderExistsError) {
                closeBusyDialog();
                showNewDialog(await document.l10n.formatValue('order-exists-error',
                    {orderNo: order.order_no}));
                return
            } else if (e instanceof UnexpectedResponseError) {
                showErrorDialog(await document.l10n.formatValue('unexpected-response-error'),
                    e.message);
            } else {
                throw(e);
            }
        } finally {
            closeBusyDialog();
        }
        document.forms['new-order'].reset();
    });
}

function showNewDialog(message) {
    let newDialog = document.querySelector('#new-dialog');
    let dialogMessage = newDialog.querySelector('.form-message');
    if (typeof message != 'undefined') {
        dialogMessage.textContent = message;
        dialogMessage.classList.add('display-form-message');
    } else {
        dialogMessage.textContent = '';
        dialogMessage.classList.remove('display-form-message');
    }
    document.forms['new-order']['created_date'].valueAsDate =
        document.forms['new-order']['preferred_date'].valueAsDate =
        document.forms['new-order']['confirmed_date'].valueAsDate =
        new Date();
    newDialog.showModal();
}

function initEditDialog(collection) {
    let editDialog = document.querySelector('#edit-dialog');
    dialogPolyfill.registerDialog(editDialog);

    for (let dropArea of editDialog.querySelectorAll('.drop-area')) {
        registerFileDropArea(dropArea);
    }

    editDialog.addEventListener('click', e => {
        if (e.target.type === 'reset') {
            editDialog.close('cancel');
        }
    });

    editDialog.addEventListener('close', async e => {
        if (editDialog.returnValue === 'cancel') {
            return;
        }

        let editForm = document.forms['edit-order'];
        let orderURL = editForm.formAction;
        let order = collection.get(orderURL);
        order.comment = editForm['comment'].value;
        order.preferred_date = editForm['preferred_date'].valueAsDate;
        order.confirmed_date = editForm['confirmed_date'].valueAsDate;
        let demandPlanDocument = editForm['demand_plan_document'].files[0];
        let confirmationDocument = editForm['confirmation_document'].files[0];
        showBusyDialog("Submitting changes…");
        try {
            await collection.updateOrder(orderURL, order, demandPlanDocument,
                confirmationDocument);
        } catch(e) {
            if (e instanceof UnexpectedResponseError) {
                showErrorDialog(await document.l10n.formatValue('unexpected-response-error'),
                    e.message);
            } else if (e instanceof NonexistentOrderError) {
                console.log(e);
            } else {
                throw(e);
            }
        } finally {
            closeBusyDialog();
        }
        editForm.formAction = '';
        editForm.reset();
    });
}

function showEditDialog(order) {
    let editDialog = document.querySelector('#edit-dialog');
    let formElement = document.forms['edit-order'];
    formElement.formAction = order.links.self;
    formElement['comment'].value = order.comment
    formElement['created_date'].value =
        getISODateString(order.created_date);
    formElement['preferred_date'].value =
        getISODateString(order.preferred_date);
    formElement['confirmed_date'].value =
        getISODateString(order.confirmed_date);
    editDialog.querySelector('#edit-order-no').textContent = order.order_no;
    editDialog.showModal();
}

function initDeleteDialog(collection) {
    let deleteDialog = document.querySelector('#delete-dialog');
    dialogPolyfill.registerDialog(deleteDialog);

    deleteDialog.addEventListener('click', e => {
        if (e.target.type === 'reset') {
            deleteDialog.close('cancel');
            return;
        }
    });

    deleteDialog.addEventListener('close', async e => {
        if (deleteDialog.returnValue === 'cancel') {
            return;
        }

        let orderURL = document.forms['delete-order'].formAction;
        document.forms['delete-order'].formAction = '';
        document.forms['delete-order'].reset();
        showBusyDialog(await document.l10n.formatValue('delete-order-busy-message'));
        try {
            await collection.deleteOrder(orderURL);
            closeBusyDialog();
        } catch (e) {
            if (e instanceof UnexpectedResponseError) {
                showErrorDialog(await document.l10n.formatValue('unexpected-response-error'),
                    e.message);
            } else if (e instanceof NonexistentOrderError) {
                console.log(e);
            } else {
                throw(e);
            }
        } finally {
            closeBusyDialog();
        }
    });
}

async function showDeleteDialog(order) {
    let deleteDialog = document.querySelector('#delete-dialog');
    document.forms['delete-order'].formAction = order.links.self;
    let dialogMessage = deleteDialog.querySelector('.dialog-message');
    document.l10n.setAttributes(dialogMessage, 'delete-order-question',
        {orderNo: order.order_no});
    deleteDialog.showModal();
}

function initViewDialog() {
    let viewDialog = document.querySelector('#view-dialog');
    dialogPolyfill.registerDialog(viewDialog);

    viewDialog.addEventListener('close', e => {
        viewDialog.querySelector('#document-viewer').remove();
    });
}

function createDocumentViewerElement(documentURL, title) {
    let template = document.querySelector('#document-viewer-template');
    let clone = template.content.cloneNode(true);

    clone.querySelector('#document-viewer').data = documentURL;

    let fallbackLink = clone.querySelector('#document-viewer > a');
    fallbackLink.href = documentURL;
    fallbackLink.title = title;
    fallbackLink.textContent = title;

    return clone;
}

function showViewDialog(order, documentType, title) {
    let viewDialog = document.querySelector('#view-dialog');
    document.l10n.setAttributes(viewDialog.querySelector('h1'),
        'view-dialog-header', {orderNo: order.order_no, title});

    let documentViewerNode =
        createDocumentViewerElement(order.links[documentType], title);
    viewDialog.querySelector('h1').after(documentViewerNode);

    viewDialog.showModal();
}

function init() {
    let collection = new OrderCollection(new URL('/api/orders',
        document.location));

    initOrdersView(collection);
    initBusyDialog();
    initErrorDialog();
    initNewDialog(collection);
    initEditDialog(collection);
    initDeleteDialog(collection);
    initViewDialog();

    collection.addEventListener('refresh', e => {
        refreshOrdersView(e.orders);
    })
    collection.refresh();
}

init();
