//
// Copyright (C) 2020 Guido Berhoerster <guido+ordertracker@berhoerster.name>
//

// +build devel

package ordertracker

import (
	"net/http"
	"os"
	"path/filepath"
)

type StaticHandler struct {
	ot *OrderTracker
}

func NewStaticHandler(ot *OrderTracker) *StaticHandler {
	return &StaticHandler{ot: ot}
}

func (h *StaticHandler) getFile(name string) http.HandlerFunc {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		f, err := os.Open(name)
		if err == os.ErrNotExist {
			h.ot.Logger.RequestDebugf(r, "%s does not exist", name)
			http.NotFound(w, r)
			return
		} else if err != nil {
			h.ot.Logger.RequestDebugf(r, "failed to open %s: %s",
				name, err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
		defer f.Close()

		d, err := f.Stat()
		if err != nil {
			h.ot.Logger.RequestDebugf(r,
				"failed to stat file %s: %s", name, err)
			http.Error(w,
				http.StatusText(http.StatusInternalServerError),
				http.StatusInternalServerError)
			return
		}
		if !d.Mode().IsRegular() {
			h.ot.Logger.RequestDebugf(r, "%s is not a regular file",
				name)
			http.NotFound(w, r)
			return
		}

		http.ServeContent(w, r, name, d.ModTime(), f)
	})
}

func (h *StaticHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	var handler http.Handler

	pwd, err := os.Getwd()
	if err != nil {
		h.ot.Logger.RequestInfof(r,
			"could not determine static directory: %s", err)
		http.Error(w, http.StatusText(http.StatusInternalServerError),
			http.StatusInternalServerError)
		return
	}

	if filename, ok := StaticDevelPaths[r.URL.Path[1:]]; ok {
		handler = h.getFile(filename)
	} else {
		staticPath := filepath.Join(pwd, "static")
		handler = http.FileServer(http.Dir(staticPath))
	}

	handler.ServeHTTP(w, r)
}
